<?php
/* --------------------------------------------------------------
   RequestAuthenticationService.php 2020-04-16
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2019 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Api\Application\Auth;

use Gambio\Api\Application\Auth\Interfaces\WebRequestAuthenticationService;
use Gambio\Api\Application\Auth\Interfaces\WebRequestAuthenticator;
use Gambio\Core\Auth\Exceptions\AuthenticationException;
use Gambio\Core\Auth\UserId;
use Gambio\Core\Permission\Model\PermissionDefinition;
use Gambio\Core\Permission\Permission;
use Gambio\Core\Permission\Repositories\PermissionsRepository;
use Psr\Http\Message\ServerRequestInterface;
use Slim\Interfaces\RouteInterface;
use Slim\Routing\RouteContext;

/**
 * Class RequestAuthenticationService
 *
 * @package Gambio\Core\Auth
 */
class RequestAuthenticationService implements WebRequestAuthenticationService
{
    /**
     * @var PermissionsRepository
     */
    private $permissionRepository;
    
    /**
     * @var WebRequestAuthenticator[]
     */
    private $authenticators;
    
    
    /**
     * AuthenticationService constructor.
     *
     * @param PermissionsRepository     $permissionRepository
     * @param WebRequestAuthenticator[] $authenticators
     */
    public function __construct(
        PermissionsRepository $permissionRepository,
        WebRequestAuthenticator ...$authenticators
    ) {
        $this->permissionRepository = $permissionRepository;
        $this->authenticators       = $authenticators;
    }
    
    
    /**
     * @inheritDoc
     */
    public function authenticateWebRequest(ServerRequestInterface $request): bool
    {
        /** @var RouteInterface $route */
        $route = $request->getAttribute(RouteContext::ROUTE);
        if ($route === null) {
            return false;
        }
        
        foreach ($this->authenticators as $authenticator) {
            try {
                $userId     = $authenticator->authenticateWebRequest($request);
                $permission = $this->getPermission($request->getMethod(), $route, $userId);
                
                return $permission->isGranted();
            } catch (AuthenticationException $exception) {
                // try next authenticator
            }
        }
        
        return false;
    }
    
    
    /**
     * @inheritDoc
     */
    public function addAuthenticator(WebRequestAuthenticator $authenticator): WebRequestAuthenticationService
    {
        $this->authenticators[] = $authenticator;
        
        return $this;
    }
    
    
    /**
     * @param string         $method
     * @param RouteInterface $route
     * @param UserId         $userId
     *
     * @return Permission
     */
    private function getPermission(string $method, RouteInterface $route, UserId $userId): Permission
    {
        switch (strtoupper($method)) {
            case 'DELETE':
                $action = 'delete';
                break;
            case 'POST':
            case 'PUT':
            case 'PATCH':
                $action = 'write';
                break;
            case 'GET':
            default:
                $action = 'read';
        }
        
        $permissionDefinition = PermissionDefinition::create($route->getPattern(), 'route', $action);
        
        return $this->permissionRepository->getPermission($userId, $permissionDefinition);
    }
}